; Test for UART1 transmitter at FLAG-0 (must be run from RAM).
; Send repeatedly the pattern 0xa5 with 8N2.
; FLAG-0 is inverted, which was a common convention for SC/MP systems due
; to their discrete RS232 driver:
; Idle is FLAG-0 high and RS232 negative (mark)
; Start bit is FLAG-1 low and RS232 positive (space)
;
; The code origins from Kitbug and must run in RAM.
;
; DLY: 4 * (13 + 2*AC + 2*DLY + 512*DLY) clock cycles
;
; 1200 baud 8N2 @ 4 MHz SC/MP II (tested):
;
; PUTC stop bit: 0x25 0x03 = 1,63 ms
; PUTC 1 bit time: 0x50 0x01 = 0.687 ms = 1455 bit/s
;
; The bit rate is always a tiny little bit larger to take loop overhead
; into account.

low function n,n&0xff
high function n,n>>8

; Delay for stop bit
stop_ac equ 0x25
stop_dly equ 0x03
; Delay for one bit
bit_ac equ 0x50
bit_dly equ 0x01

  org 0x0

  nop
  ldi low(wspace)       ; setup p2 as working space pointer
  xpal p2
  ldi high(wspace)
  xpah p2

loop:
  ldi low(uart1putc-1)
  xpal p3
  ldi high(uart1putc-1)
  xpah p3
  ldi 0xa5
  xppc p3
  jmp loop

wspace:
bitcnt equ 0
  db 0                  ; bit count for read and write
bitval equ 1
  db 0                  ; output bit

uart1putc:
  xae                   ; Save character in E
  ldi stop_ac           ; Stop bit delay
  dly stop_dly
  csa                   ; Set output to logic 0
  ori 1                 ; for start bit. (Note inversion)
  cas
  ldi 9                 ; Initialize bit count (start and 8 data bits)
  st bitcnt(p2)
uart1putbit:
  ldi bit_ac            ; Delay one bit time
  dly bit_dly
  dld bitcnt(p2)        ; Decrement bit count
  jz uart1putdone
  lde                   ; Prepare next bit
  ani 1
  st bitval(p2)
  xae                   ; Shift data right 1 bit
  sr
  xae
  csa                   ; Set up output bit
  ori 1
  xor bitval(p2)        ; Invert output bit
  cas                   ; Put bit to tty
  jmp uart1putbit
uart1putdone:
  csa                   ; Set output to logic 1
  ani 0xfe              ; for stop bit.  (Note inversion)
  cas
  xppc p3
