/* 2x M3x8 mm screws with countersunk head
 * or 2x M3x6 mm screws with cylinder head
 * SMA female connector
 * Neon bulb Philips S10 fluorescent light starter
 * DCDC converter
 * USB-C breakout board
 */

use <case_shell.scad>
use <sma905.scad>

module neon_bulb(dia,length,wire_dist,wire_dia)
{
  rotate([0,90,0]) cylinder(d=dia, h=length, $fn=36);
  
  translate([-20,0,-wire_dist/2]) rotate([0,90,0]) cylinder(d=wire_dia, h=20+dia, $fn=12);
  translate([-20,0,+wire_dist/2]) rotate([0,90,0]) cylinder(d=wire_dia, h=20+dia, $fn=12);
}

module integrating_sphere_half(dia,wall_xy,wall_z)
{
  epsilon = 0.001;
  difference()
  {
    cylinder(d=dia+2*wall_xy, h=dia/2+wall_z,$fn=36);
    translate([0,0,dia/2+wall_z])
      sphere(d=dia, $fn=360);
  }
}

module sma_holes()
{
  depth = 20;

  /* M2 core hole
   */
  sma905_screwpos()
    cylinder(d=1.6, h=depth, $fn=90);

  /* Ferrule has 3.175 mm diameter, connector is specified at
   * 3.180 mm diameter, but 3D printing is not that precise.
   * Use 0.10 mm tolerance and adapt as needed.
   */
  cylinder(d=3.18+0.10, h=depth, $fn=90);
}

module integrating_sphere_lower()
{
  dia = 33-1.5;

  difference()
  {
    union()
    {
      integrating_sphere_half(dia,0.8,0);
      /* Prevent hole from neon bulb cutout. More wall thickness
       * interferes with the screw cutouts and wastes material.
       */
      translate([dia/2,-16/2,0]) cube([1.2,16,dia/2]);
    }

    /* A little bit inside, illuminated electrode down close to the
     * SMA 905 port. The wire is actually not as thick, but the solder
     * spot is.
     */
    translate([-dia/2+1.0,0,dia/2-4])
      neon_bulb(13,30,4.5,1.5);
  }

  translate([-dia/2+1.0,0,dia/2-4])
    %neon_bulb(13,30,4.5,1.5);
}

module integrating_sphere_upper()
{
  dia = 33-1.6;

  integrating_sphere_half(dia,0.8,0);
}

module dcdc_converter()
{
  tolerance = 0.5;

  /* PCB 31x25.5 mm. Height is specified as 12, but solder spots and
   * cables do need more space.
   */
  translate([0,-(31+tolerance)/2,0])
    cube([25.5+tolerance,31+tolerance,18]);
}

module usbc_breakout(internal=false)
{
  pcb_width = 12.2;
  tolerance = internal?0.3:0.0;
  cutout = internal?10:0;

  /* PCB
   */
  translate([-(pcb_width+tolerance)/2,0,0])
    cube([pcb_width+tolerance,18+tolerance,2.1+tolerance]);

  /* USB connector. A cube for the soldering pins of the shield and
   * a rounded cube otherwise.
   */
  translate([-(9+tolerance)/2,0,2]) cube([9+tolerance,8+tolerance,3.5/2]);
  hull()
  {
    translate([-(9-2)/2,-cutout,3]) rotate([-90,0,0]) cylinder(d=2+tolerance, h=8+tolerance+cutout, $fn=36);
    translate([+(9-2)/2,-cutout,3]) rotate([-90,0,0]) cylinder(d=2+tolerance, h=8+tolerance+cutout, $fn=36);
    translate([-(9-2)/2,-cutout,4]) rotate([-90,0,0]) cylinder(d=2+tolerance, h=8+tolerance+cutout, $fn=36);
    translate([+(9-2)/2,-cutout,4]) rotate([-90,0,0]) cylinder(d=2+tolerance, h=8+tolerance+cutout, $fn=36);
  }
}

module case_base()
{
  square([67,33]);
}

studs=[[3.5,29.5,true],[63.5,3.5,true]];
screws=[[3.5,29.5,false],[63.5,3.5,false]];

module case_lower()
{
  epsilon = 0.001;
  height = 32/2;

  difference()
  {
    union()
    {
      translate([0,-33/2,0]) case_shell(height,1.2,true,false,studs) case_base();
      translate([0,-33/2,height]) rotate([0,90,0]) %usbc_breakout(internal=false);
      translate([0,-33/2,0]) cube([7,12,height]); // usbc stop
      translate([7,0,0]) %dcdc_converter();
      translate([50.0,1,0]) integrating_sphere_lower();
    }

    /* The SMA connector should face the center of the illuminated
     * part of the electrode. It is not great to have it off center,
     * but a larger sphere would waste more light.
     */
    translate([57,0,-1.2-epsilon])
      sma_holes();

    translate([0,-33/2,height]) rotate([0,90,0]) usbc_breakout(internal=true);

    translate([35,-3,-1.2+0.2-epsilon])
      rotate([0,180,0])
        linear_extrude(0.2)
          offset(r=0.2, chamfer=true)
            text("Ne/Xe",7);
  }
}

module case_upper()
{
  height = 32/2;

  rotate([180,0,0]) mirror([0,0,1])
  difference()
  {
    union()
    {
      translate([0,-33/2,0]) case_shell(height,1.2,false,false,screws) case_base();
      translate([0,-33/2,height]) rotate([0,90,0]) %usbc_breakout(internal=false);
      translate([0,-33/2,0]) cube([6.5,12,height]); // usbc stop
      translate([50.0,1,0]) integrating_sphere_upper();
      // DCDC converter motion stop
      translate([12,-33/2,0]) cube([5,5,height+5]);
    }

    translate([0,-33/2,height]) rotate([0,90,0]) usbc_breakout(internal=true);
  }
}

//neon_bulb(13,30,4.5,0.75); // center of glow at 23 mm
//integrating_sphere_lower();
//integrating_sphere_upper();
//dcdc_converter();
//rotate([0,90,0]) usbc_breakout(internal=false);

case_lower();
translate([0,50,0]) case_upper();
