/* To be printed with a 0.25 mm nozzle, extrusion width 0.25 mm,
 * layer height 0.1 mm.
 */

use <threads.scad>;

module sma905_screwpos()
{
  translate([-4.3,-4.3,0]) children();
  translate([-4.3,+4.3,0]) children();
  translate([+4.3,-4.3,0]) children();
  translate([+4.3,+4.3,0]) children();
}

/* 12.6 mm flange (some say 12.7), 2.6 mm bores, 8.6 mm bore distance
 */
module sma905_connector_flange(height, tolerance)
{
  hull()
    sma905_screwpos()
      cylinder(d=4.0+tolerance/2, h=height, $fn=90);
}

/* 1.7 mm thick
 * Made for female SMA connector of length 9.4 mm
 * Male SMA connector is ~12.0 mm long, requiring a shim of 2.6 mm
 */
module sma905_connector_internal(bore_depth)
{
  epsilon = 0.001;
  tolerance = 0.1;

  sma905_connector_flange(1.7, tolerance);

  /* Light exit for f/2 */
  translate([0,0,-bore_depth]) cylinder(d1=3.5+bore_depth/2, d2=3.5, h=bore_depth+epsilon, $fn=90);

  /* core bore for size #2 bolts (0.086"/2.18 mm with 0.102"/2.59 mm bore
   * for normal fit) or M2.2 bolts, or more common M2.
   */
  translate([0,0,-bore_depth])
    sma905_screwpos()
      cylinder(d=1.6, h=bore_depth+epsilon, $fn=90);
}

module sma905_connector_external(bore_depth)
{
  epsilon = 0.001;
  tolerance = 0.0;

  difference()
  {
  union()
  {
  sma905_connector_flange(1.7, tolerance);
  translate([0,0,1.7])
    english_thread(diameter=1/4, threads_per_inch=36, length=1/4);
  }

  /* thread hole */
  cylinder(d=4, h=10, $fn=90);

  translate([0,0,1.7+epsilon])
  {
  /* Light exit for f/2 */
  translate([0,0,-bore_depth]) cylinder(d1=3.5+bore_depth/2, d2=3.5, h=bore_depth+epsilon, $fn=90);

  /* core bore for size #2 bolts (0.086"/2.18 mm with 0.102"/2.59 mm bore
   * for normal fit) or M2.2 bolts, or more common M2.
   */
  translate([0,0,-bore_depth])
    sma905_screwpos()
      cylinder(d=1.6, h=bore_depth+epsilon, $fn=90);
  }
  }
}

module sma905_connector(bore_depth=20, internal=false)
{
  if (internal) sma905_connector_internal(bore_depth);
  else sma905_connector_external(bore_depth);
}
