#include <assert.h>
#include <stdio.h>
#include <string>
#include <time.h>
#include <fitsio.h>
#include <libraw/libraw.h>

#define elementsof(array) ((int)(sizeof(array)/sizeof(array[0])))
#define VERSION "1.0"

int main(int argc, char *argv[])
{
  int i;
  LibRaw RawProcessor;

  if (argc < 2)
  {
    printf("Usage: rawto2dspec rawfile...\n");
    return 0;
  }

#define S RawProcessor.imgdata.sizes
#define OUTR RawProcessor.imgdata.rawparams
#define P1 RawProcessor.imgdata.idata
#define P2 RawProcessor.imgdata.other
#define P3 RawProcessor.imgdata.makernotes.common
#define C RawProcessor.imgdata.color
#define HIST_BINS 32
#define HIST_BIN(v) (int)(((v)+HIST_BINS/2)*(HIST_BINS-1)/C.maximum)
#define HIST_VAL(h) ((h)*C.maximum/(HIST_BINS-1))

  for (i=1; i<argc; ++i)
  {
    int ret;
    int width, height;
    int roi_left, roi_top, roi_width, roi_height;
    int debayer;
    unsigned short *bitmap;
    std::string outfile;
    int fits_status = 0;
    fitsfile *out;
    long naxis12[2], origin[2];
    int row;
    char buf[64];
    std::string instrument;
    struct tm *tm;
    double modified_julian_date;
    std::string swcreate;
    long hist[HIST_BINS];

    if ((ret = RawProcessor.open_file(argv[i])) != LIBRAW_SUCCESS)
    {
      fprintf(stderr, "Cannot open %s: %s\n", argv[i], libraw_strerror(ret));
      exit(127);
    }

    if ((ret = RawProcessor.unpack()) != LIBRAW_SUCCESS)
    {
      fprintf(stderr, "Cannot unpack %s: %s\n", argv[i], libraw_strerror(ret));
      exit(127);
    }

    if (RawProcessor.imgdata.idata.filters < 1000)
    {
      fprintf(stderr, "Unable to process filter value %d.\n", RawProcessor.imgdata.idata.filters);
      exit(127);
    }

    if (RawProcessor.imgdata.idata.colors > 4)
    {
      fprintf(stderr, "Unable to process %d colors.\n", RawProcessor.imgdata.idata.colors);
      exit(127);
    }

    debayer = RawProcessor.imgdata.idata.colors > 1;

    outfile = '!';
    outfile += argv[1];
    outfile += ".fits";

    /* 2D spectrum images contain a narrow spectrum strip near the
     * center and the rest of the image is black and can be used
     * for background correction, which provides much less noise
     * than the few masked pixels.
     */

    width = RawProcessor.imgdata.sizes.raw_width;
    assert(width>0);
    height = RawProcessor.imgdata.sizes.raw_height;
    assert(height>0);
    roi_left = RawProcessor.imgdata.sizes.left_margin;
    roi_top = RawProcessor.imgdata.sizes.top_margin;
    roi_width = RawProcessor.imgdata.sizes.width;
    roi_height = RawProcessor.imgdata.sizes.height;
    assert(roi_left+roi_width<=width);
    assert(roi_top+roi_height<=height);
    bitmap = RawProcessor.imgdata.rawdata.raw_image;
    for (int h=0; h<HIST_BINS; ++h) hist[h] = 0;

    fits_create_file(&out, outfile.c_str(), &fits_status);
    if (fits_status) out = (fitsfile*)0; /* Bug in cfitsio 3.37 requires that. */

    if (debayer)
    {
      naxis12[0] = roi_width-1;
      roi_height &= ~1; /* unlikely, but skip odd last row */
      naxis12[1] = (roi_height/2);
    }
    else
    {
      naxis12[0] = roi_width;
      naxis12[1] = roi_height;
    }

    fits_create_img(out, FLOAT_IMG, elementsof(naxis12), naxis12, &fits_status);

    /* Rows are stored with the origin at the bottom, not at the top. That is
     * because in math diagrams have their origin there. There are some
     * programs that do not implement this, leading to images being flipped,
     * and some of them later added a flag to flip the image for the decision
     * between being bug-compatible or correct. Nevertheless, the standard is
     * clear and references
     * https://www.aanda.org/articles/aa/pdf/2002/45/aah3859.pdf which reads:
     *
     * "5.1. Image display conventions: It is very helpful to adopt a
     * convention for the display of images transferred via the FITS format.
     * Many of the current image processing systems have converged upon such a
     * convention. Therefore, we recommend that FITS writers order the pixels
     * so that the first pixel in the FITS file (for each image plane) be the
     * one that would be displayed in the lower-left corner (with the first
     * axis increasing to the right and the second axis increasing upwards)
     * by the imaging system of the FITS writer. This convention is clearly
     * helpful in the absence of a description of the world coordinates. It
     * does not preclude a program from looking at the axis descriptions
     * and overriding this convention, or preclude the user from requesting a
     * different display. This convention also does not excuse FITS writers
     * from providing complete and correct descriptions of the image
     * coordinates, allowing the user to determine the meaning of the image.
     * The ordering of the image for display is simply a convention of
     * convenience, whereas the coordinates of the pixels are part of the
     * physics of the observation."
     *
     * In order to deal with broken software, a new keyword is commonly used
     * to indicate the row order: ROWORDER = 'TOP-DOWN' or 'BOTTOM-UP'.
     */

    origin[0] = 1;

    if (debayer)
    {
      assert(!(roi_height & 1));
      for (row=roi_height-2; row>=0; row-=2)
      {
        float *buffer;
        unsigned short *cur; /* cfitsio API forbids const here */
        int col;

        buffer = new float[roi_width-1];

        cur = bitmap+(row+roi_top)*width+roi_left;
        for (col=0; col<roi_width-1; ++col)
        {
          /* Average two rows into one, but slide an average over two
           * columns. We want to keep resolution in the dispersion axis,
           * but not in the cross dispersion axis. This debayers any
           * 2x2 pattern.
           */
          buffer[col] = (cur[col]+cur[col+1]
                         +cur[col+width]+cur[col+1+width])/4.0f;
          ++hist[HIST_BIN(cur[col])];
          ++hist[HIST_BIN(cur[col+1])];
          ++hist[HIST_BIN(cur[col+width])];
          ++hist[HIST_BIN(cur[col+1+width])];
        }
        origin[1] = (roi_height - row)/2;
        assert(origin[1] >= 1);
        assert(origin[1] <= roi_height/2);
        fits_write_pix(out, TFLOAT, origin, roi_width-1, buffer, &fits_status);

        delete[] buffer;
      }
    }
    else
    {
      for (row=roi_height-1; row>=0; --row)
      {
        float *buffer;
        unsigned short *cur; /* cfitsio API forbids const here */
        int col;

        buffer = new float[roi_width];

        cur = bitmap+(row+roi_top)*width+roi_left;
        for (col=0; col<roi_width; ++col)
        {
          buffer[col] = cur[col];
          ++hist[HIST_BIN(cur[col])];
        }
        origin[1] = roi_height - row;
        assert(origin[1] >= 1);
        assert(origin[1] <= roi_height);
        fits_write_pix(out, TFLOAT, origin, roi_width, buffer, &fits_status);

        delete[] buffer;
      }
    }

    for (int h=0; h<HIST_BINS; h+=4)
    {
      char s[80];

      snprintf(s, sizeof(s), "rawto2dspec histogram %d:%ld %d:%ld %d:%ld %d:%ld\n",
             HIST_VAL(h),hist[h],
             HIST_VAL(h+1),hist[h+1],
             HIST_VAL(h+2),hist[h+2],
             HIST_VAL(h+3),hist[h+3]);
      fits_write_history(out, s, &fits_status);
    }

    fits_write_key_str(out, "ROWORDER", "BOTTOM-UP", NULL, &fits_status);

    fits_write_key_lng(out, "SNAPSHOT", 1, NULL, &fits_status);
    fits_write_key_unit(out, "SNAPSHOT", "Images", &fits_status);

    fits_write_key_str(out, "BUNIT", "adu", "Unit of primary array", &fits_status);

    fits_write_key_lng(out, "DATAMAX", (long)C.maximum, NULL, &fits_status);

#if 0
    /* not sure what of this is standardized with FITS */
      if (C.cblack[0] != 0)
      {
      fits_write_key_lng(out, "PEDESTAL", (long)C.cblack[0], NULL, &fits_status);
      fits_write_key_unit(out, "PEDESTAL", "Black level", &fits_status);

      fits_write_key_lng(out, "PEDESTA2", (long)C.cblack[1], NULL, &fits_status);

      fits_write_key_lng(out, "PEDESTA2", (long)C.cblack[2], NULL, &fits_status);

      fits_write_key_lng(out, "PEDESTA2", (long)C.cblack[3], NULL, &fits_status);
      }

      if (C.linear_max[0] != 0)
      {
      fits_write_key_lng(out, "LINMAX", (long)C.linear_max[0], NULL, &fits_status);
      fits_write_key_unit(out, "LINMAX", "Max value where still linear", &fits_status);

      fits_write_key_lng(out, "LINMAX2", (long)C.linear_max[1], NULL, &fits_status);

      fits_write_key_lng(out, "LINMAX3", (long)C.linear_max[2], NULL, &fits_status);

      fits_write_key_lng(out, "LINMAX4", (long)C.linear_max[3], NULL, &fits_status);
      }
#endif

    fits_write_key_flt(out, "EXPTIME", P2.shutter,
                       4 /* fractional mantissa digits */, NULL, &fits_status);
    fits_write_key_unit(out, "EXPTIME", "s", &fits_status);

    fits_write_date(out, &fits_status);

    /* Is the timestamp really in UTC, always?
     */
    tm = gmtime(&P2.timestamp);
    strftime(buf, sizeof(buf), "%Y-%m-%dT%H:%M:%S", tm);
    fits_write_key_str(out, "DATE-OBS", buf, "Start date of observations", &fits_status);

    /* This is expected by some FITS code. Calculate and store as double,
     * float is not enough.
     */
    modified_julian_date = 2440587.5 + (P2.timestamp/(24.0*60.0*60.0)) - 2400000.5;
    fits_write_key_dbl(out, "MJD-OBS", modified_julian_date,
                       16 /* fractional mantissa digits */, "MJD of data start time", &fits_status);

    if (P3.SensorTemperature>-999)
    {
      fits_write_key_flt(out, "CCD-TEMP", P3.SensorTemperature,
                         2 /* fractional mantissa digits */, NULL, &fits_status);
      fits_write_key_unit(out, "CCD-TEMP", "sensor temperature C", &fits_status);
    }
    else if (P3.CameraTemperature>-999)
    {
      fits_write_key_flt(out, "CCD-TEMP", P3.CameraTemperature,
                         2 /* fractional mantissa digits */, NULL, &fits_status);
      fits_write_key_unit(out, "CCD-TEMP", "camera temperature C", &fits_status);
    }

    fits_write_key_lng(out, "GAIN", (long)P2.iso_speed, NULL, &fits_status);
    fits_write_key_unit(out, "GAIN", "ISO value", &fits_status);

    instrument = "Spectrometer";
    if (P1.make[0])
    {
      instrument += " ";
      instrument += P1.make;
    }
    if (P1.model[0])
    {
      instrument += " ";
      instrument += P1.model;
    }
    if (P1.software[0])
    {
      instrument += " ";
      instrument += P1.software;
    }
    fits_write_key_str(out, "INSTRUME", instrument.c_str(), NULL, &fits_status);

    fits_write_key_str(out, "TELESCOP", RawProcessor.imgdata.lens.Lens, NULL, &fits_status);

    /* While camera format tells the physical sensor size, there is no agreement
     * if the format includes or excludes masked pixels, which forbids to calculate
     * the actual pixel size to set PIXSIZE1 and PIXSIZE2.
     */

    swcreate = "rawtospecfits ";
    swcreate += VERSION;
    swcreate += " libraw ";
    swcreate += LibRaw::version();
    swcreate += " cfitsio ";
    swcreate += std::to_string(CFITSIO_MAJOR);
    swcreate += ".";
    swcreate += std::to_string(CFITSIO_MINOR);
    swcreate += ".";
    swcreate += std::to_string(CFITSIO_MICRO);
    fits_write_key_str(out, "SWCREATE", swcreate.c_str(), NULL, &fits_status);

    fits_write_key_str(out, "HISTORY", "libraw.unpack", NULL, &fits_status);

    if (debayer)
      fits_write_key_str(out, "HISTORY", "row pair averaging, sliding column pair averaging", NULL, &fits_status);

    fits_close_file(out, &fits_status);

    if (fits_status) fits_report_error(stderr, fits_status);
  }

  return 0;
}
