include <parameters.scad>

grating_front = 4;
grating_back = 16;

// Slide frames are specified in DIN 108 with 50x50x3.2 mm.
// Real slide frames may not be as precise and have a slight
// oversize.
//
// The gratings are often not cut perpendicular, but at a
// slight angle. While 0.3 mm oversize would suffice, 3.0 mm
// oversize allows to tilt the grating slightly to compensate
// for that.

slide_size = 53.0;
slide_thickness = 3.4;

module at_grating_base_center()
{
  translate([0,grating_front,0])
    rotate([0,0,-grating_angle])
      children();
}

module at_grating_base_camera_mount()
{
  at_grating_base_center()
    rotate([0,0,-grating_angle])
      translate([0,grating_back,0])
        children();
}

module grating()
{
  cube([slide_size,slide_thickness,slide_size]);
  /* light cutout 3 mm larger than visible film */
  d=3;
  translate([7-d,-20,(13-d)]) cube([slide_size-2*(7-d),20*2,slide_size-2*(13-d)]);
}

module grating_bracket_core(width)
{
  wall = 1.6;

  // Optical axis 35 mm over ground. Subtracting base ends up at:
  raise=35-slide_size/2;

  epsilon=0.001;

    /* left pole */
    translate([-(slide_size/2)-wall,-width/2,0]) cube([wall,width,37+raise]);
    /* right pole */
    translate([+(slide_size/2),-width/2,0]) cube([wall,width,37+raise]);

    difference()
    {
      /* front and back */
      translate([-(slide_size/2),-(2*wall+slide_thickness)/2,0]) cube([slide_size,2*wall+slide_thickness,37+raise]);
      /* cutout */
      translate([-(slide_size/2),-slide_thickness/2,raise+epsilon]) grating();
    }
}

module grating_bracket(standalone=false)
{
  width = 12;

  if (standalone)
  {
    /* step for mounting screws */
    difference()
    {
      translate([-(100-6.4)/2,-width/2,0]) cube([(100-6.4),width,3]);

      translate([-(100-6.4)/2+5,0,-epsilon]) cylinder(d=2.5, h=20, $fn=12);
      translate([+(100-6.4)/2-5,0,-epsilon]) cylinder(d=2.5, h=20, $fn=12);
    }
    grating_base_core(width);
  }
  else
  {
    at_grating_base_center()
      grating_bracket_core(width);
  }
}

module grating_base()
{
  angle = 2*grating_angle;
  center_grating = [0,grating_front];
  center_camera_mount = center_grating
                        +[sin(angle)*grating_back,cos(angle)*grating_back];
  camera_mount_angle = 90-angle;

  back = (body_width/2-center_camera_mount[0])/tan(camera_mount_angle);

  camera_mount_corner1 = [body_width/2,center_camera_mount[1]-back];
  camera_mount_corner2 = 2*center_camera_mount-camera_mount_corner1;

  corners = [[-body_width/2,0], [+body_width/2,0],
             camera_mount_corner1,camera_mount_corner2];
  polygon(corners);
}
