/* 2x M3x8 mm screws with countersunk head
 * or 2x M3x6 mm screws with cylinder head
 * SMA female connector
 * Neon bulb 14.8 mm x 5.8 mm
 * DCDC converter
 * USB-C breakout board
 */

use <case_shell.scad>
use <sma905.scad>

module neon_bulb(dia,length,wire_dist,wire_dia)
{
  hull()
  {
    translate([dia/2,0,0]) sphere(d=dia, $fn=36);
    translate([length-dia/2,0,0]) sphere(d=dia, $fn=36);
  }
  translate([-20,0,-wire_dist/2]) rotate([0,90,0]) cylinder(d=wire_dia, h=20+dia, $fn=12);
  translate([-20,0,+wire_dist/2]) rotate([0,90,0]) cylinder(d=wire_dia, h=20+dia, $fn=12);
}

module integrating_sphere_half(dia,wall_x,wall_y,wall_z)
{
  epsilon = 0.001;
  difference()
  {
    translate([-(dia+wall_x)/2,-(dia+wall_y)/2,0])
      cube([dia+wall_x, dia+wall_y, (dia+wall_z)/2]);
    translate([0,0,dia/2+wall_z])
      sphere(d=dia, $fn=360);
  }
}

module sma_holes()
{
  depth = 20;

  /* M2 core hole
   */
  sma905_screwpos()
    cylinder(d=1.6, h=depth, $fn=90);

  /* Ferrule has 3.175 mm diameter, connector is specified at
   * 3.180 mm diameter, but 3D printing is not that precise.
   * Use 0.10 mm tolerance and adapt as needed.
   */
  cylinder(d=3.18+0.10, h=depth, $fn=90);
}

module integrating_sphere_lower()
{
  dia = 18;

  difference()
  {
    integrating_sphere_half(dia,1,1.2,0);

    /* A little bit inside, illuminated electrode down close to the
     * SMA 905 port.
     */
    translate([-dia/2+2.5,0,dia/2-1.8])
      neon_bulb(6,15,2,0.8);
  }

  translate([-dia/2+2.5,0,dia/2-1.8])
    %neon_bulb(6,15,2,0.8);
}

module integrating_sphere_upper()
{
  integrating_sphere_half(18,1,1.2,0);
}

module dcdc_converter()
{
  tolerance = 0.5;

  /* PCB 31x25.5 mm. Height is specified as 12, but solder spots and
   * cables do need more space.
   */
  translate([0,-(31+tolerance)/2,0])
    cube([25.5+tolerance,31+tolerance,18]);
}

module usbc_breakout(internal=false)
{
  pcb_width = 12.2;
  tolerance = internal?0.3:0.0;
  cutout = internal?10:0;

  /* PCB
   */
  translate([-(pcb_width+tolerance)/2,0,0])
    cube([pcb_width+tolerance,18+tolerance,2.1+tolerance]);

  /* USB connector. A cube for the soldering pins of the shield and
   * a rounded cube otherwise.
   */
  translate([-(9+tolerance)/2,0,2]) cube([9+tolerance,8+tolerance,3.5/2]);
  hull()
  {
    translate([-(9-2)/2,-cutout,3]) rotate([-90,0,0]) cylinder(d=2+tolerance, h=8+tolerance+cutout, $fn=36);
    translate([+(9-2)/2,-cutout,3]) rotate([-90,0,0]) cylinder(d=2+tolerance, h=8+tolerance+cutout, $fn=36);
    translate([-(9-2)/2,-cutout,4]) rotate([-90,0,0]) cylinder(d=2+tolerance, h=8+tolerance+cutout, $fn=36);
    translate([+(9-2)/2,-cutout,4]) rotate([-90,0,0]) cylinder(d=2+tolerance, h=8+tolerance+cutout, $fn=36);
  }
}

module case_base()
{
  square([52,31]);
}

studs=[[3.5,27.5,true],[48.5,3.5,true]];
screws=[[3.5,27.5,false],[48.5,3.5,false]];

module case_lower()
{
  epsilon = 0.001;
  height = 9;

  difference()
  {
    union()
    {
      translate([0,-31/2,0]) case_shell(height,1.2,true,false,studs) case_base();
      translate([0,-31/2,height]) rotate([0,90,0]) %usbc_breakout(internal=false);
      translate([0,-31/2,0]) cube([7,12,height]); // usbc stop
      translate([7,0,0]) %dcdc_converter();
      // y 1 to avoid interference with stud/screw
      translate([43,1,0]) integrating_sphere_lower();
    }

    /* M2x6
     */
    translate([43,1,-1.2-epsilon])
      sma_holes();

    translate([0,-31/2,height]) rotate([0,90,0]) usbc_breakout(internal=true);

    translate([25,-3,-1.2+0.2-epsilon])
      rotate([0,180,0])
        linear_extrude(0.2)
          offset(r=0.2, chamfer=true)
            text("Ne",7);
  }
}

module case_upper()
{
  height = 9;

  rotate([180,0,0]) mirror([0,0,1])
  difference()
  {
    union()
    {
      translate([0,-31/2,0]) case_shell(height,1.2,false,false,screws) case_base();
      translate([0,-31/2,height]) rotate([0,90,0]) %usbc_breakout(internal=false);
      translate([0,-31/2,0]) cube([7,12,height]); // usbc stop
      // y 1 to avoid interference with stud/screw
      translate([43,1,0]) integrating_sphere_upper();
    }

    translate([0,-31/2,height]) rotate([0,90,0]) usbc_breakout(internal=true);
  }
}

//neon_bulb(6,15,2,0.75);
//integrating_sphere_lower();
//integrating_sphere_upper();
//dcdc_converter();
//rotate([0,90,0]) usbc_breakout(internal=false);

//case_lower();
translate([0,40,0]) case_upper();
