use <sma905.scad>;
use <bolts.scad>;

/* Common SMA connectors are 9.4 mm from top to bottom, which lets the
 * ferrule stick out 2.6 mm. This shim avoids touching and breaking the slit.
 * My connectors are specified as 9.4 mm, but really are only 9.3 mm long.
 *
 * Check that the total length of SMA connector and shim is no less than
 * 12.0 mm.
 */
module sma905_shim(sma_length)
{
  epsilon=0.001;
  height = 12.0 - sma_length;
  center = 0.4; /* height of centering ring */

  difference()
  {
    union()
    {
      sma905_connector_flange(height, 0.0);

      /* Centering ring to reach into 4.0 connector bore. Tight tolerance
       * for press fit.
       */
      cylinder(d=4.0-0.05,h=height+center, $fn=90);
    }

    /* Bore for bolts. 2.5 is a little much, but that's what the
     * SMA connectors use, so keep it flush.
     */
    sma905_screwpos()
      translate([0,0,-epsilon])
        cylinder(d=2.5, h=height+2*epsilon, $fn=90);

    /* Ferrule has 3.175 mm diameter, connector is specified at
     * 3.180 mm diameter, but 3D printing is not that precise.
     * Use 0.10 mm tolerance and adapt as needed.
     */
    translate([0,0,-epsilon])
      cylinder(d=3.18+0.10, h=height+center+2*epsilon, $fn=90);
  }
}

module slit()
{
  epsilon=0.001;

  /* 0.2 mm tolerance to prevent screws touching the slit blades
   * leaves razor blade pieces of 5.8 mm height. Their width is
   * 6.2 mm or less (12.6/2 - 0.1 tolerance and slit width).
   *
   * For easier handling, about 5x5 mm blade pieces are best.
   */
  height = 8.6-2.6-0.2;

  /* Regular razor blades are 0.1 mm thick
   */
  translate([-12.6/2,-height/2,0]) cube([12.6,height,0.1+epsilon]);

  /* Slit orientation mark for assembly.
   */
  translate([-0.05,-12.6/2,0]) cube([0.1,12.6,0.1+epsilon]);
}

module slit_bracket_screwpos(height)
{
  holes = 4;

  for (i=[0:holes-1])
  {
    radius = 15;
    a = i*360/holes;
    translate([cos(a)*radius,sin(a)*radius,height])
      children();
  }
}

/* The focal point of the slit is 2.05 mm (2.0 mm plus half a
 * razor blade thickness) above 0.
 */
module slit_bracket_internal()
{
  height = 6;
  tolerance = 0.2; /* layer inaccuracies for layer cylinders */

  union()
  {
    cylinder(d=40+tolerance, h=height, $fn=120);

    /* M3 mounting holes and tolerance for radial correction */
    slit_bracket_screwpos(height-2.5) m3_corehole(height-2.5+10);

    translate([0,0,-10]) cylinder(d=20, h=10);
  }
}

module slit_bracket_external()
{
  height = 6;

  difference()
  {
    cylinder(d=40, h=height, $fn=120);

    /* M3 mounting holes and tolerance for radial correction */
    for (a=[-2:+2])
      rotate([0,0,a])
        slit_bracket_screwpos(height-2.5)
          m3_bolthole(height-2.5+10);

    translate([0,0,2.0]) slit();
    translate([0,0,2.0+0.1]) sma905_connector_flange(20,0.1);
    translate([0,0,2.0+0.1+2.6]) sma905_connector(internal=true);
  }
}

// The slit position is at X=0
module at_slit_bracket()
{
  translate([0,2,35]) rotate([90,0,0]) children();
}

module slit_bracket(internal=false)
{
  at_slit_bracket()
  {
    if (internal)
      slit_bracket_internal();
    else
      slit_bracket_external();
  }
}

module slit_bracket_backing_plate()
{
  at_slit_bracket() translate([0,0,-3]) cylinder(d1=43,d2=43+2*5,h=5);
}
